#include "header.h"

/* **************************************************************************************************
 * This function uses a bisection search to find the target maximum sample size R*nfix
 * required by a Delayed Response GST to terminate correctly at stage na spending
 *  the required cumulative type I and type II error probabilities.
 *
 * The function returns the needed inflation factor R.
 * 
 * **************************************************************************************************/

double inflate(double *bdupper,double *bdlower,double *bdecision, double rho, double nfix,double ratio,
	       double *observed, double *recruited, char spendtype){


  double  y=0.0, z=0.0, fy=0.0, fz=0.0, mp=0.0, fmp=0.0, grad=0.0;

  /* *******************************************************************************
   *
   * bdupper   = upper boundaries of the stopping rule at interim analyses
   * bdlower   = lower boundaries of the stopping at interim analyses
   * bdecision = critical values used at decision analyses
   * rho       = parameter indexing error spending functions to be used
   * nfix      = sample size of the corresponding fixed sample test
   * ratio     = delay parameter given by Delta_t/tmax
   * observed  = observed number of responses at interim analyses
   * recruited = number of subjects recruited at interim analyses
   * spendtype = family of error spending functions to be used.
   *    
   * ********************************************************************************/

  /* Bisection search over possible sample size inflation factors. For each choice of R, we evaluate
   * whether the error spending delayed response GST terminates properly at the final analysis.
   * At the final decision analysis, we choose boundaries lK, uK to satisfy
   *
   * P(l1<Z1<u1, ... ,l(K-1)<Z(K-1)<u(K-1), Z'K >= uK; theta) = alpha - f(I(K-1)/Imax
   * P(l1<Z1<u1, ... ,l(K-1)<Z(K-1)<u(K-1), Z'K <= lK; theta) = beta - g(I(K-1)/Imax.
   *
   * The test is said to have terminated properly if uK= lK and interim boundaries lk and uk
   * have not crossed at a previous interim analysis. The function findconstants.c returns (lK-uK).
   * If (lK-uK) >0, R can be reduced; if (lK -uK) < 0, R must be increased.*/

  y = 1.0;
  z = 2.0;

  fy = findconstants(bdupper, bdlower, bdecision, y, nfix, ratio, observed,recruited, rho, spendtype);  
  fz = findconstants(bdupper, bdlower, bdecision, z, nfix, ratio, observed,recruited, rho, spendtype);

  if(fz <= 0){
    while(fz <= 0){
      z += 0.1;
      fz = findconstants(bdupper, bdlower,bdecision, z, nfix, ratio, observed,recruited, rho, spendtype);
    }
  }
 
  if((fy<=0 && fz<=0) || (fy >=0 && fz >= 0)){
    printf("we have a problem in the inflate search, (y, fy) is (%lf, %lf); (z, fz) is (%lf, %lf) \n", y, fy, z, fz);
    exit(1);
  }
  
    while( (z-y) >tol){
      mp = (y+z)/2.0;
      fmp = findconstants(bdupper, bdlower,bdecision, mp, nfix, ratio, observed,recruited, rho, spendtype);     
      if(fmp*fy >0){
	y=mp;
	fy = fmp;
      }
      else{
	z=mp;
      }  
    } 
    /* Using linear interpolation to increase accuracy for R.*/
    fy = findconstants(bdupper, bdlower,bdecision, y, nfix, ratio, observed,recruited, rho, spendtype);
    fz = findconstants(bdupper, bdlower,bdecision, z, nfix, ratio, observed, recruited, rho, spendtype);   
    grad = (fz - fy)/(z-y);
    z = y + (-fy/grad);

    return z;
}
