#include "header.h"

/* *********************************************************************************************************** 
 * This code finds and evaluates designs for error spending Delayed Response GSTs of
 *                                        H0:theta <= 0 vs theta>0 
 * as proposed in:
 *
 * Hampson and Jennison (2012) Group sequential tests for delayed responses. To appear in J. R. Statist. Soc B.
 *
 * Our error spending designs require us to specify upfront
 *
 *  type I and type II error spending functions f(.) and g(.)
 *  target information level I_max. 
 *
 * This function uses the rho-family of error spending functions - see
 * (Jennison & Turnbull, Group Sequential Methods with Applications to Clinical Trials, Chapter 7).
 *
 * We find designs according to the improved Method 1, as discussed in Section 4.1.1.
 * Suppose by interim analysis k we have observed information levels I_1,..., I_k. Then, at interim analysis k,
 * the boundary constants defining the stopping rule are chosen to statisfy:
 *
 * P{l_1 <= Z1 <= u1, ...., l_(k-1) <= Z_(k-1) <= u_(k-1), Z_k >= u_k; theta=0} = pi_k
 * P{l_1 <= Z1 <= u1, ...., l_(k-1) <= Z_(k-1) <= u_(k-1), Z_k <= l_k; theta=delta} = gamma_k
 *
 * where this ensures that if we stopped recruitment and discarded the pipeline data,
 * we would spend type I and type II error probabilities pi_k and gamma_k as required.
 * Choosing the decision constant c_k to match reversal probabilities under theta=0 preserves the
 * type I error rate spent at stage k as pi_k.
 *
 *
 * Actually we will spend less than type II error rate beta at stage k. we correct for this by
 * calculating the amount of type II error probability to be spent at stage k, as 
 *
 * gamma_k = g(I_k/I_max) - amount of type II error actually spent up to analysis (k-1).
 *
 * **************************************************************************************************************/

int main(){ 
  int ifail=0, k=0, i=0;
  double y=0.0, z=0.0, fy=0.0, fz=0.0, grad=0.0, mp=0.0, fmp=0.0, rho=0.0, error=0.0, ratio=0.0;
  double alpha = 1-type1, beta=1-type2, nfix=0.0, nmax=0.0, rtime=0.0, otime=0.0, delay=0.0;
  double inflation=0.0,  bdupper[nastar]={0}, bdlower[nastar]={0}, bdecision[nastar]={0}, observed[nastar]={0};
  double recruited[nastar]={0}, expect[5]={0};
  char tail='L', spendtype;

  FILE *ofp1;
  ofp1 = fopen("Design_summary.txt", "w");
    
  /* ***********************************************************************************
   * Input variables from file input.txt:
   *
   * inflation = sample size inflation factor for the Delayed Response GST 
   *             such that nmax = R*nfix
   * ratio     = delay parameter representing Delta_t/tmax in the notation of our paper.
   * spendtype = string indicating the family of error spending functions to be used.
   * ***********************************************************************************/

  scanf("%lf", &inflation);
  scanf("%lf", &ratio);
  scanf("%s", &spendtype);
  
  nfix = pow(2*sigma/delta, 2.0)*pow(g01faf_(&tail, &alpha, &ifail) + g01faf_(&tail, &beta, &ifail), 2.0);
  if(ifail != 0){
    printf("we have an error in nag routine g01faf in driver.c \n");
    exit(1);
  } 
  

  /* For the given delay parameter, use a bisection search to find the
   * value of rho for which the Delayed Response GST needs maximum sample size R*nfix
   * to terminate properly at stage na spending required cumulative error probabilities.*/
  if(spendtype == 'r'){
    y = 0.1;
    z = 2;    
  }
  else{
    y=-2;
    z=2;
  }

  fy = inflate(bdupper, bdlower, bdecision, y, nfix, ratio, observed,recruited, spendtype) - inflation; 
  fz = inflate(bdupper, bdlower,bdecision, z, nfix, ratio, observed, recruited, spendtype) - inflation;
  
  if((fy<=0 && fz<=0) || (fy >=0 && fz >= 0)){
    printf("we have a problem in the rho search: the initial interval needs to be widened \n");
    exit(1);
  }

  while( (z-y) >tol){
    mp = (y+z)/2.0;
    fmp = inflate(bdupper, bdlower,bdecision, mp, nfix, ratio, observed,recruited, spendtype) - inflation;
    if(fmp*fy >0){
      y=mp;
      fy = fmp;
    }
    else{
      z=mp;
    }  
  } 
  
  /* Increase the precision of the bisection search using linear interpolation.*/ 
  fy = inflate(bdupper, bdlower, bdecision, y, nfix, ratio, observed,recruited, spendtype) - inflation; 
  fz = inflate(bdupper, bdlower, bdecision, z, nfix, ratio, observed,recruited, spendtype) - inflation;
  grad = (fz - fy)/(z-y);
  rho = y + (-fy/grad);
  
  /* Call the inflate function for one last time to update test boundaries. */ 
  fz= inflate(bdupper, bdlower,bdecision, rho, nfix, ratio, observed, recruited, spendtype);
  
  /* *******************************************************************************
   * Evaluate the expected sample sizes on termination of the error spending designs
   * when tests are designed and conducted when 
   *
   * n_k = (k/K)*(1-r)*nmax and n~_k = n_k + r*nmax  
   *
   * numbers of responses are observed and recruited at interim analysis k=1,...,K.
   * *******************************************************************************/
  
  nmax = inflation*nfix;
  delay = ratio*nmax;
  for(k=1; k<= na-1;k++){
    rtime = ratio + (1 - ratio)*((double)k/na);
    otime = (1-ratio)*((double)k/na);
    recruited[k] = rtime*nmax;
    observed[k] = otime*nmax;
  }
  observed[na] = nmax;
  expectation(observed, recruited, bdupper, bdlower, bdecision, expect);
  
  
  /* Print out of test properties.*/
  
  fprintf(ofp1, "Delayed Response GST of H0:theta <= 0 vs theta>0 designed to attain type 1 error rate %lf under theta=0 \n", type1);
  fprintf(ofp1, "and power %.3f under theta=%.3f and with target maximum sample size nmax = %.3f * nfix. \n", beta, delta, inflation);
  fprintf(ofp1, "The study was designed assuming normally distributed responses with known variance: \n");
  fprintf(ofp1, "X_(Ai) ~ N(muA, %.3f) on control \n", pow(sigma,2.0));
  fprintf(ofp1, "X_(Bi) ~ N(muB, %.3f) on experimental \n", pow(sigma,2.0));
  fprintf(ofp1, "\n");
  
  fprintf(ofp1, "Designing the error spending Delayed Response GSTs for the sequence of information levels \n");
  for(k=1; k<= na-1; k++){
    fprintf(ofp1, "I_%d = %lf  and  I'_%d = %lf \n", k, observed[k]/pow(2*sigma,2.0),k, recruited[k]/pow(2*sigma,2.0));
  }
  fprintf(ofp1, "I'_%d = %lf \n", na, observed[na]/pow(2*sigma,2.0));
  fprintf(ofp1, " \n");
  fprintf(ofp1, "Setting nmax = %.3f * nfix, the required error spending functions are given by: \n", inflation);
  if(spendtype == 'r'){
    fprintf(ofp1, "f(t) = %.4f min{t^(%.3f), 1}     and g(t) = %.3f min{t^(%.3f), 1} \n", type1, rho, type2, rho);
  }
  else{
    if(rho == 0){
      fprintf(ofp1, "f(t) = %.4f * t     and g(t) = %.3f * t \n", type1, type2);
    }
    else{
      fprintf(ofp1, "f(t) = %.4f (1 - exp(%.3f * t))/(1-exp(%.3f))  and g(t) = %.4f (1 - exp(%.3f * t))/(1-exp(%.3f))  \n", type1, (-1)*rho, (-1)*rho, type2, (-1)*rho, (-1)*rho);
    }
  }
  fprintf(ofp1, "\n");
  fprintf(ofp1, "Boundaries for monitoring standardised test statistics are: \n");
  for(k=1; k<= na-1;k++){
    fprintf(ofp1, "At interim analysis %d: l_%d = %lf    u_%d = %lf.  At decision analysis %d: c_%d = %lf \n", k, k, bdlower[k], k, bdupper[k], k, bdecision[k]);
  }
  fprintf(ofp1, "At decision analysis %d: c_%d = %lf  \n", na,na, bdecision[na]);
  
  fprintf(ofp1, "\n");
  fprintf(ofp1, "The efficiencies of the test are (expressed as percentage of nfix): \n");
  for(k=1; k<=4;k++){
    fprintf(ofp1, "F%d = %lf \n",k, 100*expect[k]/nfix);
  }
  
  fclose(ofp1);
  return 0;
}
