#include "header.h"

/* *****************************************************************************************
 * This function calculates
 * P{Continue to Stage k and stop to Accept H0; theta}
 * using recursive numerical integration (see Jennison & Turnbull, 2000, Chapter 19.2).
 *
 * The routine assumes boundaries of Delayed Response GST are calculated on the standardised
 * test statistic scale.
 * *****************************************************************************************/



double type2errorstar(int k, double *bdupper,double *bdlower, double theta, double *recruited, double *observed, double *bdecision){
  double inf1=0.0,inf2=0.0,mu=0.0,diff=0.0, t2 =0.0;
  double x[n1]={0}, xgrid[n1]={0}, y[n1]={0}, y2[n2]={0}, w1[n2]={0}, w2[n2]={0}, w3[n2]={0};
  double h1[n2]={0}, h2[n2]={0}, h3[n2]={0}, z1[n2]={0}, z2[n2]={0}, z3[n2]={0};
  int i=0, j=0,mesh=0, mesh1=0,mesh2=0, index1=0, index2=0 , m1=0, m2=0, ifail=0, s=0,i1=0, i2=0, m3=0;
  char tail = 'L'; 
  

 /* *******************************************************************************
  * k         = current stage of the GST
  * bdupper   = upper boundaries of the stopping rule at interim analyses
  * bdlower   = lower boundaries of the stopping at interim analyses
  * theta     = value of unknown parameter
  * recruited = number of subjects recruited at interim analyses
  * observed  = observed number of responses at interim analyses
  * bdecision = critical values defining decision analyses    
  * ********************************************************************************/

  m1=1;
  h1[1]=1.0;
  inf1=0.0;
  
  if(k==1){ 
    /* Create a mesh efficient for integrating marginal denisty of Z1. */ 
    inf1 = observed[1]/pow(2*sigma,2);
    inf2 = recruited[1]/pow(2*sigma,2);
    mesh = 6.0*r -1;
    for(i=1;i <= mesh;i++){
      if(i<= (r-1)){
	x[i] =(theta)*sqrt(inf1) -3.0 - 4.0*log(r/i);
      }
      else if(i >= r && i <= 5*r){
	x[i] =(theta)*sqrt(inf1) -3.0 +3.0*(i-r)/(2.0*r);
      }
      else{
	x[i]= (theta)*sqrt(inf1) + 3.0 + 4.0*log(r/(6.0*r - i));
      }
    }
    
   for(i=1;i<= mesh;i++){
     y[i] = x[i];
   }

   /* Need 2 different meshes: one containing points above u1, the other containing 
    * points for l1 to -infinity. */
  
   if(bdupper[1] >= x[mesh]){
     printf("Mesh for Z1 in type2errorstar.c with theta=%lf does not cover u_1 \n", theta);
     printf("Run program again with increased mesh parameter r \n");
     x[mesh-1] = bdupper[1];
     x[mesh] = bdupper[1] + 0.1;
     index1 = mesh-1;
   }
   else if(bdupper[1] <= x[1]){
     index1 = 1;
   }
   else{
     for(i =1;i<= mesh;i++){
       if(x[i] <= bdupper[1]){
	 index1 = i;
       }
       if(x[i] <= bdupper[1] && i < mesh){
	 x[i] = bdupper[1];
       }
     }
   }
   
 	 
   if(bdlower[1] <= y[1]){
     printf("Mesh for Z1 in type2errorstar.c with theta=%lf does not cover l_1 \n", theta);
      printf("Run program again with increased mesh parameter r \n");
      y[2] = bdlower[1];
      y[1] = bdlower[1] - 0.1;
      index2 = 2;
   }
   else if(bdlower[1] >= y[mesh]){
     index2 = mesh;
   }
   else{
     for(i=mesh;i>= 1; i--){
       if(y[i] >= bdlower[1]){
	 index2 = i;
       }
       if(y[i] >= bdlower[1] && i >1){
	 y[i] = bdlower[1];
       }
     }
   }

   /* Putting grids in different meshes and creating grids for Simpsons integration. */
   
   mesh1 = mesh - index1 +1;
   mesh2 = index2;
   for(i=1;i<= mesh1;i++){
     xgrid[i] = x[i + index1 -1];
   }
   for(i=1;i<= mesh2; i++){
     y2[i] = y[i];
   }
   m1 = 2*mesh1 -1;
   m2 = 2*mesh2 -1;

   /* Calculating mid points. */
   for(i=1;i <= m1; i++){
     j = (i+1)/2;
     z1[i] = xgrid[j];
   }
   for(i=1;i <= m2; i++){
     j = (i+1)/2;
     z2[i] = y2[j];
   }
   
   for(i=2; i <= m1-1; i= i+2){
     z1[i] = (z1[i-1] + z1[i+1])/2.0;
   }
   for(i=2; i <= m2 - 1; i= i+2){
     z2[i] = (z2[i-1] + z2[i+1])/2.0;
   }  
  

   /* Calculate Simpsons rule weightings. */
   w1[1] = (1.0/6.0)*(z1[3] - z1[1]);
   w2[1] = (1.0/6.0)*(z2[3] - z2[1]);
   
   w1[m1] = (1.0/6.0)*(z1[m1] - z1[m1-2]);
   w2[m2] = (1.0/6.0)*(z2[m2] - z2[m2-2]);
 
   for(i=2;i <= m1-1;i=i+2){
     w1[i] = (4.0/6.0)*(z1[i+1] - z1[i-1]);
   }
   for(i=3;i <= m1-2;i = i+2){
     w1[i] = (1.0/6.0)*(z1[i+2] - z1[i-2]);
   }
   
   for(i=2;i <= m2-1;i=i+2){
     w2[i] = (4.0/6.0)*(z2[i+1] - z2[i-1]);
   }
   for(i=3;i <= m2-2;i = i+2){
     w2[i] = (1.0/6.0)*(z2[i+2] - z2[i-2]);
   }   

   diff = inf2- inf1;
   t2 = 0.0;
   for(i = 1; i <= m1; i++){
     mu = (sqrt(inf2)*bdecision[k] -z1[i]*sqrt(inf1) - theta*(inf2-inf1))/sqrt(inf2-inf1);
     t2 += w1[i]*(1.0/sqrt(twopi))*exp(-0.5*pow(z1[i] - theta*sqrt(inf1),2))*g01eaf_(&tail, &mu, &ifail);
     if(ifail != 0){
       printf("there has been an error in nag routine g01eaf in type2errorstar.c\n");
     }
   }

   ifail =0;
   for(i = 1; i <= m2; i++){
     mu = (sqrt(inf2)*bdecision[k] -z2[i]*sqrt(inf1) - theta*(inf2- inf1))/sqrt(inf2-inf1);
     t2 += w2[i]*(1.0/sqrt(twopi))*exp(-0.5*pow(z2[i] - theta*sqrt(inf1),2))*g01eaf_(&tail, &mu, &ifail);
     if(ifail != 0){
       printf("there has been an error in nag routine g01eaf in type2errorstar.c \n");
     }
   }  
 }
 else{
   for(s=1; s <= k-1; s++){ 
     index1=0;
     index2=0;
     mesh=6*r-1;
     mesh1=0;
     m2=0;
     inf2 = observed[s]/pow(2*sigma,2);
     
     /* Creating mesh of grid points for integrating marginal 
      * density of Zs ~ N(theta*sqrt(inf2), 1.0) */

     for(i=1;i <= mesh;i++){
       if(i<= (r-1)){
	 x[i] =(theta)*sqrt(inf2) -3.0 - 4.0*log(r/i);
       }
       else if(i >= r && i <= 5*r){
	 x[i] =(theta)*sqrt(inf2) -3.0 +3.0*(i-r)/(2.0*r);
       }
       else{
	 x[i]= (theta)*sqrt(inf2) + 3.0 + 4.0*log(r/(6.0*r - i));
       }
     }
     
     /*To integrate over (bdlower[s], bdupper[s]) cut the grid down. */

     if(bdupper[s] >=x[mesh] && ((bdlower[s] > x[1]) && (bdlower[s] < x[mesh]))){
       index1 = mesh;
       for(i=1; i <= mesh; i++){
	 if(x[i] < bdlower[s]){
	   index2 = i;
	 }
	 if(x[i] < bdlower[s] && i< mesh){
	   x[i] = bdlower[s];
	 }
       }
     }
     else if( bdlower[s] <= x[1] && ((bdupper[s] < x[mesh]) && (bdupper[s] > x[1]))){
       index2 =1;
       for(i = mesh; i>=1; i--){
	 if( x[i] >= bdupper[s]){
	   index1 = i;
	 }
	 if( x[i] > bdupper[s] && i > 1){
	   x[i] = bdupper[s];
	 }
       }
     }
     else if( bdlower[s] <=x[1] && bdupper[s] >= x[mesh]){
       index1=mesh;
       index2 = 1;
     }
     else if( (bdlower[s]<= x[1] && bdupper[s] <= x[1]) || ((bdlower[s] >= x[mesh]) && (bdupper[s] >= x[mesh]))){
       index1 = 2;
       index2 = 1;
       x[1] = bdlower[s];
       x[2] = bdupper[s];
     }
     else{
       for(i = mesh; i>=1; i--){
	 if( x[i] >= bdupper[s]){
	   index1 = i;
	 }
	 if( x[i] > bdupper[s] && i > 1){
	   x[i] = bdupper[s];
	 }
       }
       
       for(i=1; i <= mesh; i++){
	 if(x[i] < bdlower[s]){
	   index2 = i;
	 }
	 if(x[i] < bdlower[s] && i< mesh){
	   x[i] = bdlower[s];
	 }
       }
     }
     
     mesh1 = index1 - index2 + 1;
     
     /* Mapping required original mesh points into new grid called x2 */
     
     for(i=1; i<=mesh1; i++){
       xgrid[i] = x[i+ index2 - 1];
     }
     m2 = 2*mesh1-1;
     
     /* Calculate grid mid points */
     for(i=1;i <= m2; i=i+2){
       j=(i+1)/2;
       z2[i] = xgrid[j];
     }
     for(i=2; i<= m2-1;i=i+2){
       z2[i] = (z2[i-1]+z2[i+1])/2.0;
     }
     
     /* Calculate Simpsons integration weightings */
     w2[1] = (1.0/6.0)*(z2[3]-z2[1]);
     w2[m2] = (1.0/6.0)*(z2[m2]-z2[m2-2]);
     
     for(i=2; i<=m2-1; i=i+2){
       w2[i] = (4.0/6.0)*(z2[i+1]-z2[i-1]);
     }
     for(i=3;i<=m2-2;i=i+2){
       w2[i] = (1.0/6.0)*(z2[i+2]-z2[i-2]);
     }

     diff = inf2-inf1;

     /* h2 stores the vector hk and h1 stores the vector h(k-1) */
     /* z2 are a grid of points of zk values and z1 the mesh for z(k-1). */
     for(i2=1; i2 <=m2; i2=i2+1){
       h2[i2] =0.0; 
       for(i1 = 1; i1 <= m1; i1++){
	 h2[i2] += h1[i1]*w2[i2]*sqrt(inf2/(twopi*diff))*exp((-0.5/diff)*pow(z2[i2]*sqrt(inf2)-z1[i1]*sqrt(inf1)-theta*diff,2));
       }
     }
     
     /* Overwriting h(k-1) with h(k) and also z(k-1) with zk
      * ready for the next iteration of the for loop. */
     for(i=1;i<=m2;i++){
       h1[i] = h2[i];
       z1[i]= z2[i];
     }
     m1 = m2;
     inf1=inf2;
   }

   /* We are now at interim analysis k. */
   inf2 = observed[k]/pow(2*sigma,2);
   mesh = 6.0*r -1;
   for(i=1;i <= mesh;i++){
     if(i<= (r-1)){
       x[i] =(theta)*sqrt(inf2) -3.0 - 4.0*log(r/i);
     }
     else if(i >= r && i <= 5*r){
       x[i] =(theta)*sqrt(inf2) -3.0 +3.0*(i-r)/(2.0*r);
     }
     else{
       x[i]= (theta)*sqrt(inf2) + 3.0 + 4.0*log(r/(6.0*r - i));
     }
   }
   
   /*Creating meshes for integrating over Zk from (bdupper[k], +infty) and (-infty, bdlower[k]).*/
   for(i=1;i<= mesh;i++){
     y[i] = x[i];
   }

   if(bdupper[k] >= x[mesh]){
     printf("Mesh for Zk in type2errorstar.c with theta=%lf does not cover u_k \n", theta);
     printf("Run program again with increased mesh parameter r \n");
     x[mesh-1] = bdupper[k];
     x[mesh] = bdupper[k]+0.1;
     index1 = mesh-1;
   }
   else if(bdupper[k] <= x[1]){
     index1 = 1;
   }
   else{
     for(i =1;i<= mesh;i++){
       if(x[i] <= bdupper[k]){
	 index1 = i; 
       }
       if(x[i] <= bdupper[k] && i < mesh){
	 x[i] = bdupper[k];
       }
     }
   }

 
 	 
   if(bdlower[k] <= y[1]){
     printf("Mesh for Zk in type2errorstar.c with theta=%lf does not cover l_k \n", theta);
     printf("Run program again with increased mesh parameter r \n");
     y[2] = bdlower[k];
     y[1] = bdlower[k] - 0.1;
     index2 = 2;
   }
   else if(bdlower[k] >= y[mesh]){
     index2 = mesh;
   }
   else{
     for(i=mesh;i>= 1; i--){
       if(y[i] >= bdlower[k]){
	 index2 = i;
       }
       if(y[i] >= bdlower[k] && i >1){
	 y[i] = bdlower[k];
       }
     }
   }
   

   mesh1 = mesh - index1 +1;
   mesh2 = index2;
   
   for(i=1;i<= mesh1;i++){
     xgrid[i] = x[i + index1 -1];
   }
   for(i=1;i<= mesh2; i++){
     y2[i] = y[i];
   }
 
   /* Calculating grid midpoints and Simpson integration weights.*/
   m2 = 2*mesh1 -1;
   m3 = 2*mesh2 -1;
   for(i=1;i <= m2; i=i+2){
     j = (i+1)/2;
     z2[i] = xgrid[j];
   }
   for(i=1;i <= m3; i=i+2){
     j = (i+1)/2;
     z3[i] = y2[j];
   }
   for(i=2; i <= m2-1; i= i+2){
     z2[i] = (z2[i-1] + z2[i+1])/2.0;
   }
   for(i=2; i <= m3 - 1; i= i+2){
     z3[i] = (z3[i-1] + z3[i+1])/2.0;
   }  

   w2[1] = (1.0/6.0)*(z2[3] - z2[1]);
   w3[1] = (1.0/6.0)*(z3[3] - z3[1]);
   
   w2[m2] = (1.0/6.0)*(z2[m2] - z2[m2-2]);
   w3[m3] = (1.0/6.0)*(z3[m3] - z3[m3 -2]);
   
   for(i=2;i <= m2-1;i=i+2){
     w2[i] = (4.0/6.0)*(z2[i+1] - z2[i-1]);
   }
   for(i=3;i <= m2-2;i = i+2){
     w2[i] = (1.0/6.0)*(z2[i+2] - z2[i-2]);
   }   
   for(i=2;i <= m3-1;i=i+2){
     w3[i] = (4.0/6.0)*(z3[i+1] - z3[i-1]);
   }
   for(i=3;i <= m3 - 2;i = i+2){
     w3[i] = (1.0/6.0)*(z3[i+2] - z3[i-2]);
   } 

   diff = inf2 - inf1;

   for(i2=1; i2 <=m2; i2=i2+1){
     h2[i2] =0.0; 
     for(i1 = 1; i1 <= m1; i1++){
       h2[i2] += h1[i1]*w2[i2]*sqrt(inf2/(twopi*diff))*exp(-(0.5/diff)*pow(z2[i2]*sqrt(inf2)- z1[i1]*sqrt(inf1)-theta*diff,2));
     }
   }
   
   for(i2=1; i2 <=m3; i2=i2+1){
     h3[i2] =0.0; 
     for(i1 = 1; i1 <= m1; i1++){
       h3[i2] += h1[i1]*w3[i2]*sqrt(inf2/(twopi*diff))*exp((-0.5/diff)*pow(z3[i2]*sqrt(inf2)-z1[i1]*sqrt(inf1)-theta*diff,2)); 
     }
   }

   /* Updating information levels.*/
   inf1 = inf2; 
   inf2 = recruited[k]/pow(2*sigma,2);
   diff = inf2-inf1;

   /* At decision analysis k, calculating P{Continue to Stage k, Zk >=uk, Z'k <ck; theta}.*/   
   t2 = 0.0;
   for(i = 1; i <= m2; i++){
     mu = (sqrt(inf2)*bdecision[k] - z2[i]*sqrt(inf1) - theta*diff)/ sqrt(diff);
     t2 += h2[i]*g01eaf_(&tail, &mu, &ifail);
     if(ifail != 0){
       printf("there has been an error in nag routine g01eaf in type2errorstar.c \n");
     }
   }
   
   /* At decision analysis k, calculating P{Continue to Stage k, Zk <= lk, Z'k < ck; theta}.*/
   ifail =0;
   for(i = 1; i <= m3; i++){
     mu = (sqrt(inf2)*bdecision[k] - z3[i]*sqrt(inf1) - theta*diff)/sqrt(diff);
     t2 += h3[i]*g01eaf_(&tail, &mu, &ifail);
     if(ifail != 0){
       printf("there has been an error in nag routine g01eaf in type2errorstar.c \n");
     }
   }
 }

  return t2;
}
