# Check if the current polynomial (p) is univariate. If it is, we can set the 
# indeterminate value to zero. If we find a substitution value using this 
# method, then we will return it in the form [indeterminate, value]. Otherwise
# we return false
FindIndeterminatesWhichEqualZero := function(p)
  if IsUnivariateMonomial(p) then
    return [IndeterminateOfUnivariateRationalFunction(p), p*0];
  fi;
  return false;
end;

# Check if the current polynomial can be expressed in terms of other 
# indeterminates. This is done by scanning over the univariate monomials, if
# one exists with an indeterminate which is not present in any of the
# nonUnivariateMonomials, we can rearrange to find a value for that 
# indeterminate.
#
# If an indeterminate value using this method, return it in the form 
# [indeterminate, value]. Otherwise false.
FindIndeterminatesInTermsOfOtherIndeterminates := function(p)
  local monomials, univariateMonomials, nonUnivariateMonomials, monomial, i;
  # We didn't find a univariate monomial, see if we can re arrange for an
  # indeterminate
  monomials              := MonomialsOfPolynomial(p);
  univariateMonomials    := Filtered(monomials, IsUnivariateMonomialLinear);
  nonUnivariateMonomials := Filtered(monomials, x-> not IsUnivariateMonomialLinear(x));

  for monomial in univariateMonomials do
    # Check that all the non univariate monomials don't contain the current 
    # univariate monomial indeterminate
    i := IndeterminateOfUnivariateRationalFunction(monomial);
    if ForAll(nonUnivariateMonomials, m -> not IsIndeterminateContainedInMonomial(i, m)) then
      return [i, (monomial - p) / CoefficientsOfUnivariatePolynomial(monomial)[2]];
    fi;
  od;
  return false; # nothing found
end;

# Identify if the current polynomials monomials contains a univariate factor.
# If it does, return a record representing the identified factors of the 
# polynomial if the form
#
#   rec( factors: [univariateFactor, polynomial/univarateFactor ])
#
# Otherwise return false.
FindHighestCommonUnivariateFactorOfPolynomial := function(p)
  local gcd;
  gcd := Gcd(MonomialsOfPolynomial(p));
  if IsUnivariateMonomial(gcd) and Length(CoefficientsOfUnivariatePolynomial(gcd)) > 1 then
    return rec( factors:= [gcd, p/gcd] );
  fi;
  return false;
end;

# Identify if the current polynomial has more than one factor. If it does, 
# return a DuplicateFreeList of these factors in the form
#
#   rec( factors := [a,b] )
# 
# Otherwise it returns false
#
FindFactorsOfPolynomial := function(p)
  local factors;
  factors:=Factors(p);
  if Length(factors) > 1 then
    return rec( factors:=DuplicateFreeList(factors) );
  fi;
  return false;
end;

# Identify if the current polynomial contains only one monomial. If it does
# then delegate to the FindFactorsOfPolynomial method.
#
# This method has been written to ensure that the single monomials can be 
# factorized before the FindHighestCommonUnivariateFactorOfPolynomial method
# which can only find factors with polynomials with more than 2 monomials
FindFactorsOfMonomial := function(p)
  if Length(MonomialsOfPolynomial(p)) = 1 then
    return FindFactorsOfPolynomial(p);
  fi;
  return false;
end;

# Given a polynomial, apply the substitution which is in the form
#
#   [indeterminate, value]
#
# If the value of the polynomial is zero, just return as is.
ApplySubstitution:=function(polynomial, substitution)
  if not IsZero(polynomial) then
    return Value(polynomial, [substitution[1]], [substitution[2]]);
  fi;
  return polynomial;
end;

# Scan over the list of polynomials identify any indeterminates which we can
# substitute a value for. Substitute in the determined values into the 
# polyonmials. Finally return the list of the substitutions along with the 
# polynomials after the substitutions have been substituted in
#
# Perform this logic with some initial knowledge, indeterminates and their 
# corresponding values. Start looking for substitutions from polynomials[startAt]
#
# The supplied list of polynomials should already have the given substitutions
# applied before this function is called.
SubstituteIndeterminatesWithKnowledge:=function(polynomials, substitutions, startAt)
  local FindSubstitutionsUsingMethods, result;

  # Copy polynomials and substitutions so we can make modifications on these      
  polynomials   := ShallowCopy(polynomials);
  substitutions := ShallowCopy(substitutions);

  # Use the four substitution methods in order to attempt to find values for 
  # substitutions. This method will either return false, indicating that it 
  # has found a substitution but we are not finished. This function should be
  # called again to try and find more substitutions. Otherwise it will return 
  # a list of results, stating the substitutions found and polynomials
  FindSubstitutionsUsingMethods := function()
    local p, res, subMethod, results, factor;
    for subMethod in [FindIndeterminatesWhichEqualZero, 
                      FindIndeterminatesInTermsOfOtherIndeterminates,
                      FindFactorsOfMonomial,
                      FindHighestCommonUnivariateFactorOfPolynomial,
                      FindFactorsOfPolynomial] do
      for p in [startAt..Length(polynomials)] do
        # Can only get the value of a polynomial if it is non zero
        if not IsZero(polynomials[p]) then
          # Use the current sub method to find a substitution
          res := subMethod(polynomials[p]);

          if IsRecord(res) then
            # We have identified that the current polynomial has factors which
            # may be used for finding more substitutions. What we can do is:
            #  - Replace the polynomial in polynomials with current factor
            #  - Perform a SubstituteIndeterminatesWithKnowledge with the 
            #    modified polynomial list
            #  - Combine results and return
            results := [];                    # Start with an empty list
            for factor in res.factors do
              polynomials[p] := factor;       # Replace with the current factor
              # Perform a recursive call to substitute indeterminates
              Append(results, SubstituteIndeterminatesWithKnowledge(polynomials, substitutions, p));
            od;
            return results;
          elif IsList(res) then
            Add(substitutions, res); # A single substition found. Save it
            #Sub back in to all the polynomials and substitution values
            polynomials   := List(polynomials,   p-> ApplySubstitution(p, res) );
            substitutions := List(substitutions, s-> [s[1], ApplySubstitution(s[2], res)]);
            return false;
          fi;
        fi;
      od;
    od;
    return [ rec( substitutions:=substitutions, polynomials:=polynomials ) ];
  end;

  # Keep calling FindSubstitutionsUsingMethods until we get a result
  repeat
    result := FindSubstitutionsUsingMethods();
    startAt := 1; # Reset the startAt to 1;
  until IsRecordCollection(result); # Have I got to the end?
  return result;
end;

# Look for substitutions with no initial knowledge
SubstituteIndeterminates:=function(polynomials)
  # Do an initial sort of the polynomials so that those which are most likely
  # to be hard to factorize appear later
  SortBy(polynomials, p->Length(MonomialsOfPolynomial(p)));
  return SubstituteIndeterminatesWithKnowledge(polynomials, [], 1);
end;