#include "header.h"


/* This function calculates
 * P{Continue to Stage k and stop with Zk >= c;theta}
 * using recursive numerical integration (see Jennison & Turnbull, 2000, Chapter 19.2).
 *
 * The routine assumes boundaries of Delayed Response GST are calculated on the standardised
 * test statistic scale.*/


double type1error(double *observed, int k, double *bdupper,double *bdlower, double theta, double c){
  double inf1=0.0, inf2=0.0, mu=0.0, diff=0.0, t1 = 0.0;
  double x[n2]={0}, x2[n2]={0}, w[n2]={0}, z2[n2]={0}, h1[n2]={0}, h2[n2]={0}, z1[n2]={0};
  int mesh=0, mesh1=0, index1=0, index2=0, m1=0, m2=0, i1=0,i2=0,s=0, ifail=0;
  char tail = 'L';
  
  /* *******************************************************************************
   * observed  = observed number of responses at interim analyses
   * k         = current stage of the GST
   * bdupper   = upper boundaries of the stopping rule at interim analyses
   * bdlower   = lower boundaries of the stopping at interim analyses
   * theta     = value of unknown parameter
   * c         = used in P{Continue to Stage k, observe Zk >=c;theta}    
   * ********************************************************************************/

 
  if(k==1){ 
    inf1 = observed[k]/pow(2*sigma,2.0);
    mu = -(c - theta*sqrt(inf1));
    t1 = g01eaf_(&tail, &mu, &ifail);
    if(ifail != 0){
      printf("there has been an error with g01eaf in type1error.c \n"); 
    }
  }
 else{
   inf1=0;
   m1=1;
   h1[1]=1.0;
   for (s=1; s <= k-1; s++){
     index1=0;
     index2=0;
     mesh1=0;
     m2=0;
     inf2 = observed[s]/pow(2*sigma,2.0);

     /* creating mesh of grid points suitable for integrating marginal 
      * density of Zs ~ N(theta*sqrt(inf2), 1.0) */

     mesh = 6.0*r-1;     
     for(i1=1;i1 <= mesh;i1++){
       if(i1<= (r-1)){
	 x[i1] =(theta)*sqrt(inf2) -3.0 - 4.0*log(r/((double)i1));
       }
       else if(i1 >= r && i1 <= 5*r){
	 x[i1] =(theta)*sqrt(inf2) -3.0 +3.0*(i1-r)/(2.0*r);
       }
       else{
	 x[i1]= (theta)*sqrt(inf2) + 3.0 + 4.0*log(r/(6.0*r - i1));
       }
     }
     
     /* Want to integrate over the range (bdlower[s], bdupper[s]) so cut the grid down. */

     if(bdupper[s] >=x[mesh] && (bdlower[s] > x[1] && bdlower[s] < x[mesh])){
       index1 = mesh;
       for(i1=1; i1 <= mesh; i1++){
	 if(x[i1] < bdlower[s]){
	   index2 = i1;
	 }
	 if(x[i1] < bdlower[s] && i1< mesh){
	   x[i1] = bdlower[s];
	 }
       }
     }
     else if( bdlower[s] <= x[1] && (bdupper[s] < x[mesh] && bdupper[s] > x[1])){
       index2 =1;
       for(i1 = mesh; i1>=1; i1--){
	 if( x[i1] >= bdupper[s]){
	   index1 = i1;
	 }
	 if( x[i1] > bdupper[s] && i1 > 1){
	   x[i1] = bdupper[s];
	 }
       }
     }
     else if( bdlower[s] <=x[1] && bdupper[s] >= x[mesh]){
       index1=mesh;
       index2 = 1;
     }
     else if( (bdlower[s]<= x[1] && bdupper[s] <= x[1]) || (bdlower[s] >= x[mesh] && bdupper[s] >= x[mesh])){
       index1 = 2;
       index2 = 1;
       x[1] = bdlower[s];
       x[2] = bdupper[s];
     }
     else{
       for(i1 = mesh; i1>=1; i1--){
	 if( x[i1] >= bdupper[s]){
	   index1 = i1;
	 }
	 if( x[i1] > bdupper[s] && i1 > 1){
	   x[i1] = bdupper[s];
	 }
       }
       
       for(i1=1; i1 <= mesh; i1++){
	 if(x[i1] < bdlower[s]){
	   index2 = i1;
	 }
	 if(x[i1] < bdlower[s] && i1< mesh){
	   x[i1] = bdlower[s];
	 }
       }
     }
     mesh1 = index1 - index2 + 1;
     
     /*now mapping the required original mesh points into new grid called x2 */
     
     for(i1=1; i1<=mesh1; i1++){
       x2[i1] = x[i1+ index2 - 1];
     }
     m2 = 2*mesh1-1;
     
     /*Calculate mid points of grid. */
     for(i1=1;i1 <= m2; i1=i1+2){
       z2[i1] = x2[(i1+1)/2];
     }
     for(i1=2; i1<= m2-1;i1=i1+2){
       z2[i1] = (z2[i1-1]+z2[i1+1])/2.0;
     }

     /* Calculate Simpsons rule weightings */
     w[1] = (1.0/6.0)*(z2[3]-z2[1]);
     w[m2] = (1.0/6.0)*(z2[m2]-z2[m2-2]);
     
     for(i1=2; i1<=m2-1; i1=i1+2){
       w[i1] = (4.0/6.0)*(z2[i1+1]-z2[i1-1]);
     }
     for(i1=3;i1<=m2-2;i1=i1+2){
       w[i1] = (1.0/6.0)*(z2[i1+2]-z2[i1-2]);
     }
     
    
     diff = inf2-inf1;
     /* h2 stores the vector hk and h1 stores the vector h(k-1) */
     /* z2 is grid of points of Zk values and z1 the mesh for Z(k-1). */
     for(i2=1; i2 <=m2; i2=i2+1){
       h2[i2] =0.0;
       for(i1 = 1; i1 <= m1; i1++){
	 h2[i2] = h2[i2] +
	    + h1[i1]*w[i2]*sqrt(inf2/(twopi*diff))*exp((-0.5/diff)*pow(z2[i2]*sqrt(inf2)-z1[i1]*sqrt(inf1)-theta*diff,2));
       }
     }
     
     /* overwriting h(k-1) with h(k) and also z(k-1) with zk
      * ready for the next iteration of the for loop.*/
     for(i1=1;i1<=m2;i1++){
       h1[i1] = h2[i1];
       z1[i1] = z2[i1];
     }
     m1=m2;
     inf1=inf2;
   }

   /* Now we're at stage k, finish calculating P(l1<Z1<u1 ,..., l(k-1)<Z(k-1)<u(k-1), Zk>c; theta). */
   
   /*updating information levels.*/
   inf1 = observed[k-1]/pow(2*sigma,2.0);
   inf2 = observed[k]/pow(2*sigma,2.0);
   diff = inf2-inf1;

   t1 = 0.0;
   for(i1 = 1; i1<=m2; i1++){
     mu = (-c*sqrt(inf2) +z2[i1]*sqrt(inf1) + theta*diff)/sqrt(diff);
     t1 += h1[i1]*g01eaf_(&tail, &mu, &ifail);
     if(ifail !=0){
       printf("there is an error in the nag g01eaf in type1error.c \n");
     }
   }
 }

 return t1;

 }







