#include "header.h"


/******************************************************************************************************
 * This function designs error spending Delayed Response GSTs assuming interim and decision           *
 * analyses are conducted at numbers of observed responses:
 *
 *
 * n_k = (k/K)*(1-r)*nmax      and       n~_k = n_k + r*nmax 
 *
 * and information levels
 *
 * I_k = (k/K)*(1-r)*I_max     and       I~_k = I_k + r*I_max
 *
 * Information levels are calculated assuming responses are normally distributed with known variance.
 *
 *
 * The function assumes nmax = R*nfix, where R has been specified by the user, and assumes
 * the target sample size nmax will be reached at stage K in the absence of early stopping.
 *
 *
 * Given the specified information sequence, this function computes the critial values of the
 * error spending Delayed Response GST, spending error probabilities according to either the
 * rho family (Jennison & Turnbull, Chapter 7, 2000) or gamma family (Hwang, Shih & DeCani, 1990)
 * of error spending functions for one-sided tests. Error probabilities are spent as a
 * function of observed information. 
 * 
 * It may be that for the sample size inflation factor R we have set, the error spending functions
 * we have chosen (indexed by rho) do not ensure proper termination of the test at stage K,
 * i.e., choosing lK and uK to satisfy
 *
 * P{l1< Z1 <u1, ..., l(K-1)<Z(K-1)<u(K-1), Z'K > uK; theta=0} = alpha - f(I(K-1)/Imax)
 * P{l1< Z1 <u1, ..., l(K-1)<Z(K-1)<u(K-1), Z'K < lK; theta=delta} = beta - g(I(K-1)/Imax)
 *
 * results in uK != lK. This function returns the value of (lK - uK) so we can search for the value of
 * rho for which uK = lK.
 *
 ********************************************************************************************************/ 

double findconstants(double *bdupper, double *bdlower,double *bdecision, double R,double nfix,
		     double ratio, double *observed, double *recruited, double rho, char spendtype){
  int k=0;
  double spend1=0.0, spend2=0.0, frac1=0.0, frac2 = 0.0, diff=0.0, nmax=0.0, otime=0.0, rtime=0.0, cumt2=0.0;
  

  /* *******************************************************************************
   * bdupper   = upper boundaries of the stopping rule at interim analyses
   * bdlower   = lower boundaries of the stopping at interim analyses
   * bdecision = critical values at decision analyses
   * R         = sample size inflation factor with nmax = R*nfix
   * nfix      = sample size required by the fixed sample test
   * ratio     = delay parameter 
   * observed  = observed number of responses at interim analyses
   * recruited = number of subjects recruited at interim analyses
   * rho       = parameter indexing error spending functions 
   * spendtype = specifying whether error probabilities are to be spent according to
   *             rho or gamma family of functions. If spendtype=='r' use rho functions
   *             otherwise use gamma functions.    
   *********************************************************************************/

  /* specify the numbers of observed responses at each interim analysis and # subject recruited
   * assuming interim analyses equally spaced in calendar between times Delta_t and tmax, when 
   * all nmax subjects would have been recruited in the absence of early stopping.*/

  nmax = R*nfix;
  for(k=1; k<= na-1;k++){
    otime = (1-ratio)*((double)k/na);
    rtime = ratio + (1-ratio)*((double)k/na);
    observed[k] = otime*nmax;
    recruited[k] = rtime*nmax;
  }
  observed[na] = nmax;
  recruited[na] = nmax;
 
  

  cumt2 =0.0;
  for(k=1;k<=na; k++){
    frac1 = observed[k]/observed[na];
    
    if(spendtype == 'r'){
      /* Evaluate error probs to be spent at stage k using rho functions.
       * When calculating type II error prob, calculate the amount of type II
       * actually spent up to stage (k-1). */
      spend1 = type1*(min(pow(frac1, rho), 1.0) - min(pow(frac2, rho), 1.0));
      spend2 = type2*min(pow(frac1, rho), 1.0) - cumt2;
    }
    else{ 
      /* Spend error according to gamma family of functions.*/
      if(rho != 0){
	spend1 = (1-exp(-rho*min(frac1, 1.0)))/(1-exp(-rho));
	spend1 -= (1-exp(-rho*min(frac2, 1.0)))/(1-exp(-rho));
	spend1 *= type1;
	
	spend2 = type2*(1-exp(-rho*min(frac1, 1.0)))/(1-exp(-rho));
	spend2 -= cumt2; 
      }else{
	spend1 = type1*(min(frac1, 1.0) - min(frac2, 1.0));
	spend2 = type2*min(frac1, 1.0) - cumt2;
      }
    }

    /* Find the solution to the pair of equations:
     *
     * P(l1<Z1<u1,.., l(k-1)< Z(k-1)< u(k-1), Zk >uk; theta=0) = spend1,
     * P(l1<Z1<u1,.., l(k-1)< Z(k-1)< u(k-1), Z < lk; theta=delta) = spend2. 
     *
     * The first LHS probability is decreasing with increasing uk.
     * The second LHS probability is increasing with increasing lk.
     * So use bisection search to find (lk,uk) satisfying equations.*/

    if(bdlower[k-1] <= bdupper[k-1]){
      bisection(bdupper, bdlower, spend1, spend2, k,observed);
      if( (bdlower[k] <= bdupper[k]) && (k < na)){
	/* calculate decision constant and cumulative type II error probability spent up to stage k.*/
	symmetry(bdecision, bdupper,bdlower, observed, recruited, k);
	cumt2 += type2errorstar(k, bdupper, bdlower, delta, recruited, observed, bdecision);
      }else{
	bdecision[k] = bdecision[k-1];
      }
    }
    else{
      bdupper[k] = bdupper[k-1];
      bdlower[k] = bdlower[k-1];
      bdecision[k]= bdecision[k-1];
    }

    frac2 = frac1;  

  }

  diff = bdlower[na] - bdupper[na];


  return diff;
}
