%
% Wavelet transform
%
%-------------------------------Copyright----------------------------------
%
% Author: Aleksandra Pidde, aleksandra.pidde@gmail.com
% 
% Related articles:
% J. Newman, A. Pidde, A. Stefanovska, "Defining the Wavelet Bispectrum"
% {Preprint submitted to Applied and Computational Harmonic Analysis, March 5, 2019, preprint:arXiv ?}
%
%------------------------------Documentation-------------------------------
%
% [wt, freq, wavPar] = wav(x, fs, wavelet, f0, fmin, fmax, nv, p, cutEdges, logBase)
% - calculate contionous wavelet transform [wt] with the normalisation of a signals [x, y, z] sampled at [fs] Hz.
% 
% INPUT:
% x, y, z    - signals
% fs         - sampling frequency of the signals
% wavelet    - type of wavelet to choose from {'morlet', 'lognorm', 'bump'}, the most suitable and recommended is 'lognorm' 
%              wavelet
% f0         - frequency resolution parameter, (\sigma_0), typically recommended > 0.7
% fmin/fmax  - minimal/maximal frequency for which bispectrum will be calculated
% nv         - 'number of voices', frequency discretization, the next frequency equals previous one multiplied by
%              [logBase^(1/nv)]
% p          - normalising pre-factor, the most recommeded p = 1 for logarithmic frequency approach
% cutEdges   - cutting the cone of influence, {0, 1}, 1 is recommeded, everything that does not fit in 2 std of Gaussian window 
%              is cut
% logBase    - base of the logarithm for which frequency axis is generated, typically 2 is chosen but the 'logarithmic
%              frequency approach' is defined for base exp(1) so the choise of exp(1) could simplify the integration
% T          - the time window T, for which the bispectrum is averaged, [tmin tmax] (sec), where tmin <= tmax
%
% OUTPUT:
% wt         - wavelet transform
% freq:      - frequencies
% Optional:
% wavPar     - fwt: Fourier transform, f0, fpeak: numerical value for which the Fourier transform of the wavelet is maximised, 
%              base: logBase, Cphi: numerical value of formula 18;
%

function [wt, freq, wavPar] = wav(x, fs, wavelet, f0, fmin, fmax, nv, p, cutEdges, logBase)

wavPar = {};
N = length(x);
nq = ceil((N + 1) / 2); 
ff = [(0 : nq - 1), -fliplr(1 : N - nq)] * fs / N; 
freqFT = ff(:);

ftx = fft(x); % FT signal
switch wavelet
    case 'morlet'
        % Morlet
        fwt = @(f) exp(- 0.5 * (2 * pi * f0 * (f - 1)).^2) .* (1 - exp(-(2 * pi * f0).^2 * f));
        [val, ind] = max(abs(fwt(freqFT)));
        fpeak = freqFT(ind);

    case 'lognorm'
        % Lognormal 
        fwt = @(f) (f > 0) .* exp(-(2 * pi * f0 * log(f)).^2 / 2);
        [val, ind] = max(abs(fwt(freqFT)));
        fpeak = freqFT(ind);
   
    case 'bump'
        % Bump
        D = 0.4 / f0;
        fwt = @(f) (f > 1 - D) .* (f < 1 + D) .* exp(1 - (1 ./ (1 - ((f - 1) / D).^2)));
        [val, ind] = max(abs(fwt(freqFT)));
        fpeak = freqFT(ind);
    otherwise
        error('Unknown wavelet type!, choose: morlet/lognorm/bump')

end
coi = @(f) floor(2 * f0 * fs / f); % 2 sigma of Gaussian window in time domain = 2 * f0 / f in points
if fmin == fmax
    freq = fmin;
else
    freq = logBase.^((ceil(nv * log(fmin) / log(logBase)) : floor(nv * log(fmax) / log(logBase)))' / nv);
end
nf = length(freq);
wt = NaN * zeros(nf, N);

for i = 1 : nf
    i1 = 1;
    if cutEdges
        i1 = max(1, coi(freq(i)));     
    end
    if 2 * i1 < N
        n = (freq(i) / abs(fpeak)).^(p - 1);
        freqwt = (fpeak / freq(i)) * freqFT;
        phiBar = conj(fwt(freqwt)).';
        conv = n * ftx .* phiBar;
        conv(isnan(conv)) = 0;
        out = ifft(conv);
        wt(i, i1 : end - i1 + 1) = out(i1 : end - i1 + 1);
    end
    
end
wavPar.fwt = fwt; 
wavPar.f0 = f0;
wavPar.fpeak = fpeak;
wavPar.base = logBase;
nvC = 10 * nv;
fmini = 1e-6; fmaxi = 1e6;
freqC = exp((ceil(nvC * log(fmini)) : floor(nvC * log(fmaxi)))' / nvC);
wavPar.Cphi = sum(fwt(freqC).^2) / nvC;
end

